// Copyright 2009 Google Inc. All Rights Reserved.

package org.unicode.cldr.icu;

import org.unicode.cldr.icu.DeprecatedConverter.MakefileInfo;
import org.unicode.cldr.util.CLDRFile;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Calendar;
import java.util.List;
import java.util.Set;

class ICUMakefileWriter {
  private final ICULog log;

  ICUMakefileWriter(ICULog log) {
    this.log = log;
  }

  public void write(MakefileInfo info, String stub, String shortStub, File dstDir) {
    String resfiles_mk_name = dstDir + "/" + shortStub + "files.mk";

    String generatedAliasText = nameSetToList(info.generatedAliasFiles);
    String aliasFilesText = nameSetToList(info.aliasFromFiles);
    String ctdFilesText = createFileList(info.ctdFiles);
    String brkFilesText = createFileList(info.brkFiles);
    String emptyFilesText = info.emptyFromFiles.isEmpty() ? null : nameSetToList(info.emptyFromFiles);
    String inFilesText = nameSetToList(info.fromFiles);

    try {
      log.setStatus(null);
      log.log("Writing ICU build file: " + resfiles_mk_name);

      Calendar c = Calendar.getInstance();
      int year = c.get(Calendar.YEAR);
      String localmk = shortStub + "local.mk";

      PrintStream ps = new PrintStream(new FileOutputStream(resfiles_mk_name));
      ps.println("# *   Copyright (C) 1998-" + year + ", International Business Machines");
      ps.println("# *   Corporation and others.  All Rights Reserved.");
      ps.println(stub + "_CLDR_VERSION = " + CLDRFile.GEN_VERSION);
      ps.println("# A list of txt's to build");
      ps.println("# Note:");
      ps.println("#");
      ps.println("#   If you are thinking of modifying this file, READ THIS.");
      ps.println("#");
      ps.println("# Instead of changing this file [unless you want to check it back in],");
      ps.println("# you should consider creating a '" + localmk + "' file in this same directory.");
      ps.println("# Then, you can have your local changes remain even if you upgrade or");
      ps.println("# reconfigure ICU.");
      ps.println("#");
      ps.println("# Example '" + localmk + "' files:");
      ps.println("#");
      ps.println("#  * To add an additional locale to the list:");
      ps.println("#    _____________________________________________________");
      ps.println("#    |  " + stub + "_SOURCE_LOCAL =   myLocale.txt ...");
      ps.println("#");
      ps.println("#  * To REPLACE the default list and only build with a few");
      ps.println("#    locales:");
      ps.println("#    _____________________________________________________");
      ps.println("#    |  " + stub + "_SOURCE = ar.txt ar_AE.txt en.txt de.txt zh.txt");
      ps.println("#");
      ps.println("#");
      ps.println("# Generated by LDML2ICUConverter, from LDML source files.");
      ps.println();

      ps.println("# Aliases without a corresponding xx.xml file (see " +
          DeprecatedConverter.SOURCE_INFO + ")");
      ps.println(stub + "_SYNTHETIC_ALIAS =" + generatedAliasText);
      ps.println();
      ps.println();

      ps.println("# All aliases (to not be included under 'installed'), but not including root.");
      ps.println(stub + "_ALIAS_SOURCE = $(" + stub + "_SYNTHETIC_ALIAS)" + aliasFilesText);
      ps.println();
      ps.println();

      if (ctdFilesText != null) {
        ps.println("# List of compact trie dictionary files (ctd).");
        ps.println("BRK_CTD_SOURCE = " + ctdFilesText);
        ps.println();
        ps.println();
      }

      if (brkFilesText != null) {
        ps.println("# List of break iterator files (brk).");
        ps.println("BRK_SOURCE = " + brkFilesText);
        ps.println();
        ps.println();
      }

      if (emptyFilesText != null) {
        ps.println("# Empty locales, used for validSubLocale fallback.");
        ps.println(stub + "_EMPTY_SOURCE =" + emptyFilesText);
        ps.println();
        ps.println();
      }

      ps.println("# Ordinary resources");
      if (emptyFilesText == null) {
        ps.print(stub + "_SOURCE =" + inFilesText);
      } else {
        ps.print(stub + "_SOURCE = $(" + stub + "_EMPTY_SOURCE)" + inFilesText);
      }
      ps.println();
      ps.println();

      ps.close();
    } catch(IOException e) {
      log.error("While writing " + resfiles_mk_name, e);
      System.exit(1);
    }
  }

  private static final String LINESEP = System.getProperty("line.separator");

  private static String createFileList(List<String> list) {
    if (list == null || list.isEmpty()) {
      return null;
    }
    StringBuilder sb = new StringBuilder();
    for (String string : list) {
      sb.append(" ").append(string);
    }
    return sb.toString();
  }

  private static String nameSetToList(Set<String> names) {
    StringBuilder out = new StringBuilder();
    int i = 0;
    for (String name : names) {
      if ((++i % 5) == 0) {
        out.append("\\").append(LINESEP);
      }
      out.append(" ").append(name).append(".txt");
    }
    return out.toString();
  }
}