/*! \file app.c
\brief This file is part of the simple utf16 compression/decompression application.  

It reads data from standard input and writes to standard output. If it is
executed without a parameter it displays simple help and exits.

This source code is public-domain. You can use, modify, and distribute the
source code and executable programs based on the source code.

This software is provided "as is" without express or implied warranty. 
*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "utf16_compress.h"

/** 
* maximum decompressed data input length (1MB)
*/
#define MAX_DECOMPRESSED_LENGTH 1048576

/** 
* maximum compressed data input length 
*/
#define MAX_COMPRESSED_LENGTH (MAX_DECOMPRESSED_LENGTH + (MAX_DECOMPRESSED_LENGTH/2))

/**
* just a malloc wrapper
*/
static void * xmalloc(size_t size)
{
  void * ret = malloc(size);
  if (!ret)
  {
    printf("malloc failed!\n");
    exit(1);
  }
  return ret;
}

/**
* just print short help and exit 
*/
static void usage(void)
{
  fprintf(stderr, "Need argument:\n c - compress\n x - extract\n");
  exit(1);
}

/**
* reads data from stdin, compress it and writes output to stdout
*/
static void compress_stdin(void)
{
  int i, c, out_len;
  char *buffer, *output_buffer;
  
  i = 0;
  /* allocate input and output buffers */
  buffer = (void *)xmalloc(MAX_DECOMPRESSED_LENGTH);
  output_buffer = (void *)xmalloc(MAX_COMPRESSED_LENGTH);
  
  /* read input from stdin */
  while((c=getchar()) != EOF)
  { 
    if (i == MAX_DECOMPRESSED_LENGTH)
    {
      fprintf(stderr, "ERROR: Input is too long!\n");
      exit(1);
    }
    buffer[i] = c;
    i++;
  }

  /* each utf16 character is 2 bytes long so the input number of bytes must be odd */
  if (i % 2 != 0)
    fprintf(stderr, "WARNING: odd number of input bytes!\n");

  /* compress */
  out_len = utf16_compress((u16 *)buffer, i / 2, (u8 *)output_buffer);

  free(buffer);

  /* print output to stdout */
  for (i=0;i<out_len;i++)
    putchar(output_buffer[i]);

  free(output_buffer);
}

/**
* reads compressed utf16 from stdin, decompress it and writes output to stdout
*/
static void decompress_stdin(void)
{
  int i, c, out_len;
  char *buffer, *output_buffer;
  
  i = 0;

  /* allocate input and output buffers */
  buffer = (void *)xmalloc(MAX_COMPRESSED_LENGTH);
  output_buffer = (void *)xmalloc(MAX_DECOMPRESSED_LENGTH);
  
  /* read input from stdin */
  while((c=getchar()) != EOF)
  { 
    if (i == MAX_DECOMPRESSED_LENGTH)
    {
      fprintf(stderr, "ERROR: Input is too long!\n");
      exit(1);
    }
    buffer[i] = c;
    i++;
  }

  /* make decompression */
  out_len = utf16_decompress((u8 *)buffer, i, (u16 *)output_buffer);

  free(buffer);

  /* print output to stdout */
  for (i=0;i<out_len;i++)
    putchar(output_buffer[i]);

  free(output_buffer);
}


int main(int argc, char **argv)
{
  if (argc != 2 || strlen(argv[1]) != 1)
    usage();
  if (argv[1][0] == 'x')
    decompress_stdin();
  else if (argv[1][0] == 'c')
    compress_stdin();
  else 
    usage();
  return 0;
}

